/*
   This program creates two child processes and then builds
   a two stage pipeline out of those two processes. This
   parent process is not part of the pipeline. It does not
   share in any of the work done in the pipeline.

   To build the pipeline, the parent starts each of the child
   processes, then it makes the first child inherit the parent's
   stdin, then it makes the second child inherit the parent's
   stdout, then it redirects the first child's stdout to the
   second child's stdin. The parent should end up sharing stdin
   with the first child and stdout with the second child, but
   the parent should make no use of stdin and stdout while its
   children are running.


   Important: Notice that this program is logically
   the same as the following command line pipe.

   C:\> java RemoveVowels | java ToUpperCase

   In the case of the above command line, the shell
   program (cmd.exe) sets up the pipeline before running
   the two programs RemoveVowels.class and ToUpperCase.class.
   In the case of this program, the program itself creates
   the pipeline between the two child processes. So this
   program is acting like a very simple shell program.
*/
import java.util.Scanner;
import java.io.*;

public class Java6_Pipeline_ver1
{
   public static void main(String[] args) throws IOException, InterruptedException
   {
      // Create a command line for running the stage1 child.
      String cmd1 = "java RemoveVowels";
      // Execute the first filter.
      Process process1 = Runtime.getRuntime().exec(cmd1);

      // Create a command line for running the stage2 child.
      String cmd2 = "java ToUpperCase";
      // Execute the second filter.
      Process process2 = Runtime.getRuntime().exec(cmd2);


      // Get references to the streams for sending input into each stage.
      PrintStream stage1In = new PrintStream( process1.getOutputStream() );
      PrintStream stage2In = new PrintStream( process2.getOutputStream() );

      // Get references to the streams for reading output from each stage.
      Scanner stage1Out = new Scanner( process1.getInputStream() );
      Scanner stage2Out = new Scanner( process2.getInputStream() );


      // Create a Scanner object to make it easier to use System.in
      Scanner scanner = new Scanner( System.in );

      // Send all of the data from stdin to the first stage of the pipeline.
      while ( scanner.hasNextLine() )
      {
         String oneLine = scanner.nextLine();  // read from stdin
         stage1In.println( oneLine );          // write to first stage
      }
      stage1In.close(); //THIS IS IMPORTANT (otherwise process2 will hang)


      // Send all of the data from the first stage into the second stage of the pipeline.
      while ( stage1Out.hasNextLine() )
      {
         String oneLine = stage1Out.nextLine();  // read from first stage
         stage2In.println( oneLine );            // write to second stage
      }
      stage2In.close(); //THIS IS IMPORTANT (otherwise the parent will hang)


      // Now read all of the data out of the second stage and send it to stdout.
      while ( stage2Out.hasNextLine() )
      {
         String oneLine = stage2Out.nextLine();
         System.out.println( oneLine );
      }

      // Wait for the second child to finish its work.
      process2.waitFor();  // this throws InterruptedException
   }
}